import { VercelRequest, VercelResponse } from '@vercel/node';
import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import rateLimit from 'express-rate-limit';

// Import your existing routes
import menuRoutes from '../backend/src/routes/menu.js';
import authRoutes from '../backend/src/routes/auth.js';
import restaurantRoutes from '../backend/src/routes/restaurants.js';
import imageRoutes from '../backend/src/routes/images.js';
import csvRoutes from '../backend/src/routes/csv.js';
import brandTemplateRoutes from '../backend/src/routes/brandTemplate.js';
import performanceRoutes from '../backend/src/routes/performance.js';

const app = express();

// Security middleware
app.use(helmet());
app.use(cors({
  origin: process.env.CORS_ORIGIN || "*",
  credentials: true
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100 // limit each IP to 100 requests per windowMs
});
app.use(limiter);

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true }));

// API routes
app.use('/api/auth', authRoutes);
app.use('/api/restaurants', restaurantRoutes);
app.use('/api', menuRoutes);
app.use('/api', imageRoutes);
app.use('/api', csvRoutes);
app.use('/api', brandTemplateRoutes);
app.use('/api/performance', performanceRoutes);

// Health check endpoint
app.get('/api/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date().toISOString() });
});

// Export the Express app as a Vercel function
export default (req: VercelRequest, res: VercelResponse) => {
  return app(req, res);
};